<?php

function optimizeAndSaveImage(string $sourcePath, string $destinationPath, int $maxWidth = 1920, int $quality = 85): bool
{
    if (!file_exists($sourcePath) || !is_readable($sourcePath)) {
        error_log("Optimizer Error: El archivo fuente no existe o no se puede leer: " . $sourcePath);
        return false;
    }

    $imageInfo = getimagesize($sourcePath);
    if ($imageInfo === false) {
        error_log("Optimizer Error: No se pudo obtener información de la imagen: " . $sourcePath);
        return false;
    }
    
    $imageType = $imageInfo[2];
    $originalWidth = $imageInfo[0];
    $originalHeight = $imageInfo[1];

    try {
        $image = null;
        switch ($imageType) {
            case IMAGETYPE_JPEG:
                $image = imagecreatefromjpeg($sourcePath);
                break;
            case IMAGETYPE_PNG:
                $image = imagecreatefrompng($sourcePath);
                break;
            case IMAGETYPE_GIF:
                error_log("Optimizer Warning: GIF no procesado a WebP.");
                return false; 
            case IMAGETYPE_WEBP:
                $image = imagecreatefromwebp($sourcePath);
                break;
            default:
                error_log("Optimizer Error: Tipo de imagen no soportado para optimización: " . $imageType);
                return false;
        }

        if (!$image) {
            error_log("Optimizer Error: No se pudo crear el recurso de imagen desde: " . $sourcePath);
            return false;
        }

        $newWidth = $originalWidth;
        $newHeight = $originalHeight;

        if ($originalWidth > $maxWidth) {
            $ratio = $originalWidth / $originalHeight;
            $newWidth = $maxWidth;
            $newHeight = (int)($newWidth / $ratio);
        }

        $newImage = imagecreatetruecolor($newWidth, $newHeight);

        if ($imageType == IMAGETYPE_PNG || $imageType == IMAGETYPE_WEBP) {
            imagealphablending($newImage, false);
            imagesavealpha($newImage, true);
        }
        
        imagecopyresampled($newImage, $image, 0, 0, 0, 0, $newWidth, $newHeight, $originalWidth, $originalHeight);
        
        $destinationDir = dirname($destinationPath);
        if (!is_dir($destinationDir)) {
            mkdir($destinationDir, 0755, true);
        }

        $success = imagewebp($newImage, $destinationPath, $quality);

        imagedestroy($image);
        imagedestroy($newImage);

        return $success;

    } catch (Exception $e) {
        error_log("Optimizer Exception: " . $e->getMessage());
        return false;
    }
}
?>