<?php
// Incluimos la conexin desde la misma carpeta 'admin'
// Asegrate de que esta ruta sea CORRECTA. Si db_admin-connection.php est en el mismo directorio, es 'db_admin-connection.php'.
// Si est en un directorio diferente, ajusta la ruta.
require_once 'db_admin-connection.php'; 

// Configuraciones de error para depuracin: Mantenemos esto para que puedas ver cualquier error fatal.
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Indicamos que la respuesta ser en formato JSON
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");

// Obtener los parmetros de paginacin y bsqueda de la solicitud GET
$limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 6;
$offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
$searchTerm = isset($_GET['search_term']) ? trim($_GET['search_term']) : ''; // Nuevo: trmino de bsqueda

// Asegrate de que la variable $pdo est disponible globalmente.
// Si db_admin-connection.php establece $pdo como una variable global, esto es necesario.
global $pdo; 

try {
    // VERIFICACIN CRTICA: Aseguramos que $pdo es un objeto PDO vlido ANTES de usarlo.
    // Esto es para capturar errores si db_admin-connection.php falla en establecer la conexin.
    if (!isset($pdo) || !$pdo instanceof PDO) {
        throw new Exception("Error interno: La conexin a la base de datos no est disponible. Verifique 'db_admin-connection.php'.");
    }

    // Construir la clusula WHERE para la bsqueda
    $whereClause = '';
    $bindParams = []; // Usaremos un array para bindParam, ms seguro y flexible

    if (!empty($searchTerm)) {
        // Convertimos tanto la columna de la DB como el trmino de bsqueda a minsculas
        // para una bsqueda insensible a maysculas/minsculas.
        $whereClause = " WHERE LOWER(title) LIKE LOWER(:searchTerm) OR LOWER(description) LIKE LOWER(:searchTerm) ";
        $bindParams[':searchTerm'] = '%' . $searchTerm . '%';
    }

    // Primero, obtener el nmero total de videos que coinciden con el trmino de bsqueda
    $stmtCount = $pdo->prepare("SELECT COUNT(*) AS total FROM academy" . $whereClause);
    if (!empty($bindParams)) {
        foreach ($bindParams as $param => $value) {
            $stmtCount->bindValue($param, $value, PDO::PARAM_STR);
        }
    }
    $stmtCount->execute();
    $totalVideos = $stmtCount->fetch(PDO::FETCH_ASSOC)['total'];

    // Leemos la tabla 'academy' con LIMIT, OFFSET y la clusula WHERE
    $query = "SELECT title, description, url AS video_url, poster_url FROM academy" . $whereClause . " ORDER BY created_at DESC LIMIT :limit OFFSET :offset";
    $stmt = $pdo->prepare($query);
    
    if (!empty($bindParams)) {
        foreach ($bindParams as $param => $value) {
            $stmt->bindValue($param, $value, PDO::PARAM_STR);
        }
    }
    $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    
    $videos = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Preparar la respuesta incluyendo los videos y el total
    $response = [
        'videos' => $videos,
        'total' => $totalVideos,
        'current_count' => $offset + count($videos),
        'debug_query' => $query // Opcional: para depuracin, puedes ver la query ejecutada
    ];

    echo json_encode($response);

} catch (PDOException $e) {
    http_response_code(500);
    // Registra el error completo en el log para una depuracin detallada
    error_log("Error PDO en admin_academy.php: " . $e->getMessage() . " - Query: " . (isset($query) ? $query : 'N/A') . " - SearchTerm: " . $searchTerm);
    echo json_encode(['error' => 'Error de conexin o consulta de base de datos. Por favor, revise los logs del servidor.', 'message' => $e->getMessage()]);
} catch (Exception $e) {
    http_response_code(500);
    // Registra el error completo en el log para una depuracin detallada
    error_log("Error inesperado en admin_academy.php: " . $e->getMessage());
    echo json_encode(['error' => 'Error inesperado en el servidor. Por favor, revise los logs del servidor.', 'message' => $e->getMessage()]);
}