<?php
ob_start();
require_once __DIR__ . '/../config.php';
ob_clean();

header('Content-Type: application/json');

try {
    // Verificación de API Key
    if (!defined('GEMINI_API_KEY') || empty(GEMINI_API_KEY)) {
        http_response_code(500);
        echo json_encode(['error' => 'La clave de API de Gemini no está configurada.']);
        exit;
    }
    $apiKey = GEMINI_API_KEY;

    // Verificación del método HTTP
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Método no permitido.']);
        exit;
    }

    // Obtención y validación de datos
    $json_data = file_get_contents('php://input');
    $data = json_decode($json_data, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        http_response_code(400);
        echo json_encode(['error' => 'Formato de datos JSON inválido.']);
        exit;
    }

    // Sanitizar y validar las entradas
    $user_name = filter_var($data['user_name'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $value_proposition = filter_var($data['value_proposition'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $company_name = filter_var($data['company_name'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $client_name = filter_var($data['client_name'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $service_type = filter_var($data['service_type'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $scenario = filter_var($data['scenario'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $message_tone = filter_var($data['message_tone'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $additional_details = filter_var($data['additional_details'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $contact_phone = filter_var($data['contact_phone'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $contact_web = filter_var($data['contact_web'] ?? '', FILTER_SANITIZE_URL);

    // Validar campos obligatorios
    if (empty($user_name) || empty($company_name) || empty($client_name) || empty($service_type) || empty($scenario) || empty($message_tone)) {
        http_response_code(400);
        echo json_encode(['error' => 'Faltan campos obligatorios.']);
        exit;
    }

    // Mapeo de escenarios a descripciones para el prompt
    $scenario_descriptions = [
        'satisfaction' => [
            'purpose' => "un mensaje de agradecimiento y confirmación de satisfacción para un cliente (después de 1 día). El objetivo es consolidar la buena experiencia y abrir un canal de comunicación para cualquier comentario.",
            'tip' => "Este consejo debe enfocarse en cómo usar este mensaje para consolidar la buena impresión inicial, invitando al cliente a compartir su experiencia. Sugiere maneras de fomentar la confianza y la lealtad desde el principio."
        ],
        'recurrent' => [
            'purpose' => "un mensaje de recordatorio de mantenimiento o para agendar el próximo servicio (después de 3-5 días o cuando sea recurrente). El mensaje debe ser un recordatorio suave y estratégico para fomentar la recompra del servicio.",
            'tip' => "Este consejo debe orientar sobre cómo este mensaje puede servir como un recordatorio del valor a largo plazo del servicio, sin presionar para una venta. Debe centrarse en mantener la relación y la marca presente en la mente del cliente."
        ],
        'review' => [
            'purpose' => "un mensaje para solicitar amablemente una reseña o testimonio. Debe ser un mensaje de agradecimiento, preguntar si el cliente quedó satisfecho y proporcionar un enlace directo para que deje su reseña, facilitando el proceso.",
            'tip' => "Este consejo debe resaltar la importancia de las reseñas para la reputación online. Sugiere personalizar el mensaje, agradecer al cliente por la confianza y explicar de manera concisa por qué su opinión es valiosa para tu negocio."
        ],
        'referral' => [
            'purpose' => "un mensaje para solicitar referidos a un cliente satisfecho. El mensaje debe ser una invitación amable a recomendar tu servicio a sus amigos y familiares, mencionando un beneficio para ambas partes (ej. un descuento).",
            'tip' => "Este consejo debe enfocarse en cómo aprovechar la confianza de un cliente satisfecho. Recomienda ser transparente con los beneficios del programa de referidos y facilitar el proceso para que el cliente pueda compartir la información fácilmente."
        ],
        'reengage' => [
            'purpose' => "un mensaje de reconexión para clientes que no han contratado un servicio en 3-6 meses. El objetivo es recordarles tu negocio, ofrecer un incentivo para volver y preguntar de forma genuina si necesitan algo.",
            'tip' => "El consejo debe centrarse en cómo reactivar a clientes pasados de forma no intrusiva. Sugiere ofrecer un pequeño descuento o un servicio extra para motivar el retorno y mantener la relación."
        ],
        'anniversary' => [
            'purpose' => "un mensaje personalizado para felicitar al cliente por el aniversario de su primera contratación. El mensaje debe ser de agradecimiento y celebrar la relación que han construido. Es puramente para fidelización.",
            'tip' => "El consejo debe orientar sobre la importancia de la personalización para construir lealtad. Recomienda que el mensaje sea genuino y no intente vender nada, sino simplemente celebrar la relación con el cliente."
        ]
    ];

    $purpose = $scenario_descriptions[$scenario]['purpose'];
    $tip_purpose = $scenario_descriptions[$scenario]['tip'];

    // Construcción del CTA basado en los datos proporcionados
    $cta_block = "";
    if (!empty($contact_phone) && !empty($contact_web)) {
        $cta_block = "\n\n¡Estamos aquí para lo que necesites!\n📞 Contáctanos: {$contact_phone}\n🌐 Visita nuestra página: {$contact_web}\n";
    } elseif (!empty($contact_phone)) {
        $cta_block = "\n\n¡Estamos aquí para lo que necesites!\n📞 Contáctanos: {$contact_phone}\n";
    } elseif (!empty($contact_web)) {
        $cta_block = "\n\n¡Estamos aquí para lo que necesites!\n🌐 Visita nuestra página: {$contact_web}\n";
    }

    // Construcción del PROMPT MAESTRO con el nuevo CTA y el tono
    $prompt = "Actúa como un experto en redacción de mensajes para clientes existentes de negocios de servicios locales. Tu objetivo es generar 3 ideas de mensajes diferentes, persuasivos y empáticos que refuercen la lealtad y fomenten la recompra.

**CONTEXTO CLAVE:**
- Tu Nombre (el remitente): \"{$user_name}\"
- Nombre de Tu Negocio: \"{$company_name}\"
- Nombre del Cliente (el destinatario): \"{$client_name}\"
- Servicio Realizado: \"{$service_type}\"
- Escenario: {$purpose}
- Tono Solicitado: {$message_tone}
" . (!empty($additional_details) ? "- Detalles Adicionales: \"{$additional_details}\"\n" : "") . "
" . (!empty($contact_phone) ? "- Teléfono de Contacto: \"{$contact_phone}\"\n" : "") . "
" . (!empty($contact_web) ? "- Página Web/Social: \"{$contact_web}\"\n" : "") . "

**TU TAREA:**
Genera 3 sugerencias de mensajes diferentes. Para cada sugerencia, genera:
1.  **spanish_message (Mensaje en Español):** Un mensaje conciso para el cliente (máximo 5-7 líneas). Debe ser natural e incluir el nombre del cliente y del negocio. **Si se proporcionó información de contacto, finaliza el mensaje con un bloque de contacto apropiado usando las etiquetas 'Contáctanos:' y 'Visita nuestra página:'.**
2.  **english_message (Mensaje en Inglés):** La traducción precisa del `spanish_message` al inglés. **Si el mensaje en español tiene un bloque de contacto, traduce las etiquetas a 'Contact us:' y 'Visit our page:'.**
3.  **tip (Consejo Estratégico):** Un consejo práctico para el dueño del negocio sobre la estrategia detrás de este tipo de mensaje. **Este consejo debe ser conciso, accionable y redactado ÚNICAMENTE en español**. {$tip_purpose}.

**REGLAS IMPORTANTES:**
-   **NO uses marcadores de posición** como `[Nombre del Cliente]` en los mensajes finales.
-   **NO incluyas texto introductorio o explicaciones** fuera del objeto JSON.
-   La salida debe ser un **objeto JSON válido** que contenga un array llamado `suggestions`. Este array debe contener 3 objetos, cada uno con las claves \"spanish_message\", \"english_message\" y \"tip\".
";

    // Llamada a la API de Google Gemini con esquema forzado
    $apiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.5-flash:generateContent?key=' . $apiKey;

    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => [
            'response_mime_type' => 'application/json',
            'response_schema' => [
                'type' => 'OBJECT',
                'properties' => [
                    'suggestions' => [
                        'type' => 'ARRAY',
                        'items' => [
                            'type' => 'OBJECT',
                            'properties' => [
                                'spanish_message' => ['type' => 'STRING'],
                                'english_message' => ['type' => 'STRING'],
                                'tip' => ['type' => 'STRING']
                            ],
                            'required' => ['spanish_message', 'english_message', 'tip']
                        ],
                        'minItems' => 3,
                        'maxItems' => 3
                    ]
                ],
                'required' => ['suggestions']
            ]
        ]
    ];

    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_TIMEOUT => 90
    ]);

    $response_ai = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    // ==================================================================================
    // CAMBIO CLAVE: Manejo de errores mejorado y logging
    // ==================================================================================

    // Si la respuesta de la API de Gemini falla, registra el error y devuelve un JSON claro.
    if ($response_ai === false || $http_code !== 200) {
        http_response_code($http_code === 0 ? 503 : 502);
        $error_details = $curl_error ?: ($response_ai ? json_decode($response_ai, true) : 'Respuesta de la API vacía o ilegible.');
        error_log("Error de conexión/respuesta de IA en api-asistente-de-seguimiento-estrategico.php: HTTP Code " . $http_code . ", cURL Error: " . $curl_error . ", Details: " . json_encode($error_details, JSON_UNESCAPED_UNICODE));
        echo json_encode([
            'error' => 'No se pudo obtener una respuesta válida del servicio de IA.',
            'details' => $error_details,
            'http_code' => $http_code
        ]);
        exit;
    }

    // Registra la respuesta cruda de la IA para depuración
    error_log("Respuesta cruda de la IA: " . $response_ai);

    $result_array = json_decode($response_ai, true);
    $response_text = $result_array['candidates'][0]['content']['parts'][0]['text'] ?? null;
    
    if ($response_text === null) {
        http_response_code(500);
        error_log("Formato de respuesta de IA inesperado. Clave 'text' no encontrada. Respuesta cruda: " . json_encode($result_array, JSON_UNESCAPED_UNICODE));
        echo json_encode([
            'error' => 'La IA devolvió un formato de respuesta inesperado. Falta la clave de texto generada.',
            'raw_ai_response' => $result_array,
        ]);
        exit;
    }

    $response_text = str_replace(['```json', '```'], '', $response_text);
    $final_data = json_decode(trim($response_text), true);

    if ($final_data === null || !isset($final_data['suggestions']) || !is_array($final_data['suggestions']) || count($final_data['suggestions']) !== 3) {
        http_response_code(500);
        error_log("Formato de respuesta de IA inválido en api-asistente-de-seguimiento-estrategico.php. Respuesta cruda: " . trim($response_text) . ", JSON Error: " . json_last_error_msg());
        echo json_encode([
            'error' => 'La IA devolvió un formato de JSON inválido o no contiene los campos esperados.',
            'raw_ai_response_after_trim' => trim($response_text),
            'decoded_data_debug' => $final_data,
            'json_last_error' => json_last_error_msg()
        ]);
        exit;
    }

    // Si todo es correcto, limpiar el buffer y enviar la respuesta.
    ob_end_clean();
    echo json_encode($final_data, JSON_UNESCAPED_UNICODE);
    exit;

} catch (Exception $e) {
    http_response_code(500);
    error_log("Excepción interna: " . $e->getMessage());
    echo json_encode(['error' => 'Error interno del servidor: ' . $e->getMessage()]);
    exit;
} finally {
    if (ob_get_level() > 0) {
        ob_end_flush();
    }
}
?>